unit fROR_VistARegistries;
{
================================================================================
*
*       Package:        ROR - Clinical Case Registries
*       Date Created:   $Revision: 10 $  $Modtime: 3/06/07 2:17p $
*       Site:           Hines OIFO
*       Developers:
*                                      
*
*       Description:    VistA Registry Selector
*
*       Notes:
*
================================================================================
*       $Archive: /CCR v1.5/Current/fROR_VistARegistries.pas $
*
* $History: fROR_VistARegistries.pas $
 * 
 * *****************  Version 10  *****************
 * User: Vhaishgavris Date: 3/08/07    Time: 1:47p
 * Updated in $/CCR v1.5/Current
 * 
 * *****************  Version 9  *****************
 * User: Vhaishgavris Date: 3/02/05    Time: 11:54a
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 8  *****************
 * User: Vhaishgavris Date: 1/24/05    Time: 1:31p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 7  *****************
 * User: Vhaishgavris Date: 1/10/05    Time: 3:48p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 6  *****************
 * User: Vhaishgavris Date: 12/06/04   Time: 3:54p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 5  *****************
 * User: Vhaishgavris Date: 10/14/04   Time: 3:50p
 * Updated in $/CCR v1.0/Current
 * 
 * *****************  Version 4  *****************
 * User: Vhaishgavris Date: 6/24/04    Time: 3:13p
 * Updated in $/CCR v1.0/Current
 *
 * *****************  Version 3  *****************
 * User: Vhaishgavris Date: 4/15/04    Time: 3:55p
 * Updated in $/CCR v1.0/Current
 *
*
================================================================================
}

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  uROR_Utilities, StdCtrls, uROR_PatientDataLists, uROR_Common,
  uROR_Errors, Buttons, ExtCtrls, uROR_Reports;

type

  TRegistryInfo = class(TPersistent)
  private
    fAdmin:               Boolean;
    fAutoConfirm:         Boolean;
    fAvailableReports:    TAvailableReports;
    fDescription:         String;
    fIEN:                 String;
    fInactivationReasons: TSetOfCodes;
    fInvestDrugClasses:   TStringList;
    fLoaded:              Boolean;
    fName:                String;
    fSupportingEvidences: TSetOfCodes;

    function getAvailableReports: TAvailableReports;
    function getInactivationReasons: TSetOfCodes;
    function getSupportingEvidences: TSetOfCodes;

  public
    constructor Create;
    destructor  Destroy; override;

    procedure AssignRawData(RawData: String);
    procedure AssignTo(aDest: TPersistent); override;
    function  Load(Reload: Boolean = False): TReturnCode; virtual;

    property Admin: Boolean                       read fAdmin;
    property AutoConfirm: Boolean                 read fAutoConfirm;
    property AvailableReports: TAvailableReports  read getAvailableReports;
    property Description: String                  read fDescription;
    property IEN: String                          read fIEN;
    property InactivationReasons: TSetOfCodes     read getInactivationReasons;
    property InvestDrugClasses: TStringList       read fInvestDrugClasses;
    property Name: String                         read fName;
    property SupportingEvidences: TsetOfCodes     read getSupportingEvidences;

  end;

  TAvailableRegistries = class(TStringList)
  private
    fLoaded: Boolean;

    function  getInsertLocation(aReg: TRegistryInfo): integer;
    function  getObject(anIndex: Integer): TRegistryInfo; reintroduce;
    procedure InsertBlank;
    procedure setObject(anIndex: Integer; aValue: TRegistryInfo);

  public
    constructor Create;
    destructor  Destroy; override;

    function  Add(const RawData: String): Integer; override;
    procedure AssignTo(aDest: TPersistent); override;
    procedure Clear; override;
    procedure Delete(anIndex: Integer); override;
    function  IndexOfName(const aName: String): Integer; override;
    function  Load(Reload: Boolean = False): TReturnCode;
    function  Select(const aRegName: String = '' ): TRegistryInfo;

    property Objects[anIndex: Integer]: TRegistryInfo read    getObject
                                                      write   setObject;
                                                      default;

  end;

  TFormSelectRegistry = class(TForm)
    pnlMain: TPanel;
    lbRegistries: TListBox;
    Panel1: TPanel;
    btnCancel: TBitBtn;
    btnOK: TBitBtn;
    procedure lbRegistriesDblClick(Sender: TObject);
    procedure FormCreate(Sender: TObject);
    procedure lbRegistriesClick(Sender: TObject);
    procedure FormActivate(Sender: TObject);

  published
    procedure ApplyOptions;

  end;

function AvailableRegistries: TAvailableRegistries;

implementation
{$R *.DFM}

uses
  uROR_Strings, uROR_Broker, fROR_Options, uROR_Classes;

var
  fAvailableRegistries: TAvailableRegistries = nil;

function AvailableRegistries: TAvailableRegistries;
begin
  if not Assigned(fAvailableRegistries) then
    fAvailableRegistries := TAvailableRegistries.Create;
  Result := fAvailableRegistries;
end;

{$REGION ' TAvailableRegistries '}
///////////////////////////// TAvailableRegistries \\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TAvailableRegistries.Create;
begin
  inherited Create;
  CaseSensitive := False;
  Duplicates    := dupError;
  fLoaded       := False;
  Sorted        := False;
end;

destructor TAvailableRegistries.Destroy;
begin
  Clear;
  inherited;
end;

function TAvailableRegistries.Add(const RawData: String): Integer;
var
  regInfo: TRegistryInfo;
begin
  regInfo := TRegistryInfo.Create;
  try
    regInfo.AssignRawData(RawData);
    if (regInfo.IEN = '1') or (regInfo.IEN = '2') then
      Result := AddObject(regInfo.IEN, regInfo)
    else begin
      Result := getInsertLocation(regInfo);
      if Result > Count then
        Result := AddObject(regInfo.IEN, regInfo)
      else
        InsertObject(Result, regInfo.IEN, regInfo);
    end;
  except
    regInfo.Free;
    Raise;
  end;
end;

function TAvailableRegistries.getInsertLocation(aReg: TRegistryInfo): integer;
var
  regInfo: TRegistryInfo;
  i: integer;
begin
  result := -1;
  i := Count;
  while (i > 0) and (result = -1) do
  begin
    regInfo := getObject(i-1);
    if (regInfo.IEN = '1') or (regInfo.IEN = '2') or
       (aReg.Description > regInfo.Description) then
      result := i;
    dec(i);
  end;

  if (i = 0) and (Result = -1) then
    Result := i
  else if Result = -1 then
    Result := Count + 1;
end;

procedure TAvailableRegistries.InsertBlank;
var
  regInfo: TRegistryInfo;
  i, j: integer;
  bNational: boolean;
begin
  bNational := false;
  j := -1;
  i := 0;
  while (i < Count) and (j = -1) do
  begin
    regInfo := getObject(i);
    if (regInfo.IEN = '1') or (regInfo.IEN = '2') then
      bNational := true
    else if (regInfo.IEN <> '1') and (regInfo.IEN <> '2') and (bNational)
      and (i > 0) then
      j := i;
    inc(i);
  end;

  if j <> -1 then
  begin
    regInfo := TRegistryInfo.Create;
    try
      regInfo.AssignRawData('-99^^0^^0');
      InsertObject(j, regInfo.IEN, regInfo);
    except
      regInfo.Free;
      Raise;
    end;
  end;
end;

procedure TAvailableRegistries.AssignTo(aDest: TPersistent);
var
  i, n: Integer;
begin
  if aDest is TStrings then
    begin
      TStrings(aDest).Clear;
      n := Count - 1;
      for i := 0 to n do TStrings(aDest).Add(Objects[i].Description);
    end
  else
    inherited;
end;

procedure TAvailableRegistries.Clear;
var
  i: Integer;
begin
  for i:=Count-1 downto 0 do
  begin
    try
      Objects[i].Free;
    except
    end;
    Objects[i] := nil;
  end;
  inherited;
end;

procedure TAvailableRegistries.Delete(anIndex: Integer);
begin
  try
    Objects[anIndex].Free;
    Objects[anIndex] := nil;
  except
  end;
  inherited;
end;

function TAvailableRegistries.getObject(anIndex: Integer): TRegistryInfo;
begin
  Result := inherited Objects[anIndex] as TRegistryInfo;
end;

function TAvailableRegistries.IndexOfName(const aName: String): Integer;
var
  i, n: Integer;
begin
  Result := -1;
  n := Count - 1;
  for i:=0 to n do
    if Assigned(Objects[i]) and (Objects[i].Name = aName) then
    begin
      Result := i;
      Break;
    end;
end;

function TAvailableRegistries.Load(Reload: Boolean): TReturnCode;
var
  i, n: Integer;
  rawData: TStringList;
  {$IFDEF DBUGETM6}
  DebugETM: TextFile;
  DebugETMisTrue: boolean;
  {$ENDIF}
begin
  if not fLoaded or Reload then
  begin
    Clear;
    rawData := TStringList.Create;
    try
      if Broker.CallProc(rpcGUIAccess, [], nil, rawData) then
      begin
        n := StrToIntDef(Piece(rawData[0], '^'), 0);
        {$IFDEF DBUGETM6}
        DebugETMisTrue := FileExists('C:\ETM\DEBUG-ETM-6.txt');
        if DebugETMisTrue then
        begin
          System.AssignFile(DebugETM, 'C:\ETM\DEBUG-ETM-6.txt');
          System.Append(DebugETM);
          writeln(DebugETM, '*********Begin ROR GUI ACCESS in TAvailableRegistries.Load*********');
          writeln(DebugETM, 'n=['+IntToStr(n)+']');
          writeln(DebugETM, 'rawData=['+rawData.Text+']');
          writeln(DebugETM, '*********End ROR GUI ACCESS in TAvailableRegistries.Load*********');
          System.CloseFile(DebugETM);
        end;
        {$ENDIF}
        for i := 1 to n do
          Add(rawData[i]);
        InsertBlank;
        fLoaded := True;
        Result := RC_OK;
      end
      else
        Result := RCE_RPC_ERROR;
    finally
      rawData.Free;
    end;
  end
  else
    Result := RC_OK;
end;

function TAvailableRegistries.Select(const aRegName: String): TRegistryInfo;
var
  n, resultNdx: Integer;
begin
  Result := nil;
  if not IsRCError(Load) then
  begin
    resultNdx := -1;
    n := Count - 1;
    if aRegName <> '' then
    begin
      resultNdx := IndexOfName(aRegName);
      if resultNdx < 0 then
        MessageDlg508(rscAccessViolation, Format(RSC00170, [aRegName]),
          mtError, [mbOK], 0);
    end
    else if n < 0 then
      MessageDlg508(rscAccessViolation, RSC00171, mtError, [mbOK], 0)
    else if n = 0 then
      resultNdx := 0
    else
      with TFormSelectRegistry.Create(Application) do
      begin
        Self.AssignTo(lbRegistries.Items);
        lbRegistries.ItemIndex := 0;
        if ShowModal = mrOK then
          resultNdx := lbRegistries.ItemIndex;
        Free;
      end;
    if resultNdx >= 0 then
      Result := Objects[resultNdx];
  end;
end;

procedure TAvailableRegistries.setObject(anIndex: Integer; aValue: TRegistryInfo);
begin
  inherited Objects[anIndex] := aValue;
end;
{$ENDREGION}

{$REGION ' TFormSelectRegistry '}
///////////////////////////// TFormSelectRegistry \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

procedure TFormSelectRegistry.ApplyOptions;
var
  ie: TCCRInterfaceElement;
begin
  inherited;
  ie := CCROptions.IE[ieReadOnly];
  lbRegistries.Color := ie.Color;
  lbRegistries.Font.Color := ie.FontColor;
end;

procedure TFormSelectRegistry.FormActivate(Sender: TObject);
begin
  //-- Section 508: Set focus to list of registries
  if CCRScreenReaderActive then
    lbRegistries.SetFocus;
end;

procedure TFormSelectRegistry.FormCreate(Sender: TObject);
begin
  ApplyOptions;
end;

procedure TFormSelectRegistry.lbRegistriesClick(Sender: TObject);
begin
  btnOk.Enabled := length(trim(lbRegistries.Items[lbRegistries.ItemIndex])) > 0;
  btnOk.TabStop := btnOk.Enabled;
end;

procedure TFormSelectRegistry.lbRegistriesDblClick(Sender: TObject);
begin
  if length(trim(lbRegistries.Items[lbRegistries.ItemIndex])) > 0 then
    btnOk.Click;
end;
{$ENDREGION}

{$REGION ' TRegistryInfo '}
//////////////////////////////// TRegistryInfo \\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\\

constructor TRegistryInfo.Create;
begin
  inherited;
  fLoaded := False;

  fAvailableReports    := TAvailableReports.Create;
  fInactivationReasons := TSetOfCodes.Create('1');
  fInvestDrugClasses   := TStringList.Create;
  fSupportingEvidences := TSetOfCodes.Create('2');
end;

destructor TRegistryInfo.Destroy;
begin
  FreeAndNil(fSupportingEvidences);
  FreeAndNil(fInvestDrugClasses);
  FreeAndNil(fInactivationReasons);
  FreeAndNil(fAvailableReports);
  inherited;
end;

procedure TRegistryInfo.AssignRawData(RawData: String);
begin
  fIEN         := Piece(RawData, '^', 1);
  fName        := Piece(RawData, '^', 2);
  fAdmin       := (StrToIntDef(Piece(RawData, '^', 3), 0) <> 0);
  fDescription := Piece(RawData, '^', 4);
  fAutoConfirm := (StrToIntDef(Piece(RawData, '^', 5), 0) <> 0);
end;

procedure TRegistryInfo.AssignTo(aDest: TPersistent);
var
  dst: TRegistryInfo;
begin
  if aDest is TRegistryInfo then
  begin
    dst := TRegistryInfo(aDest);

    dst.fInactivationReasons.Assign(fInactivationReasons);
    dst.fSupportingEvidences.Assign(fSupportingEvidences);

    dst.fAdmin       := fAdmin;
    dst.fAutoConfirm := fAutoConfirm;
    dst.fDescription := fDescription;
    dst.fIEN         := fIEN;
    dst.fName        := fName;
  end
  else
    inherited;
end;

function TRegistryInfo.getAvailableReports: TAvailableReports;
begin
  Load;
  Result := fAvailableReports;
end;

function TRegistryInfo.getInactivationReasons: TSetOfCodes;
begin
  Load;
  Result := fInactivationReasons;
end;

function TRegistryInfo.getSupportingEvidences: TSetOfCodes;
begin
  Load;
  Result := fSupportingEvidences;
end;

function TRegistryInfo.Load(Reload: Boolean): TReturnCode;
begin
  Result := RC_OK;
  if not fLoaded or Reload then
  begin
    if IsRCError(fInactivationReasons.Load(IEN)) then
      Result := RCE_LOAD_INACT_REASONS
    else if IsRCError(fSupportingEvidences.Load(IEN)) then
      Result := RCE_LOAD_SUP_EVIDENCES
    else if IsRCError(fAvailableReports.Load(IEN)) then
      Result := RCE_LOAD_AVAIL_REPORTS;
    fLoaded := True;
  end;
end;
{$ENDREGION}

////////////////////////////////////////////////////////////////////////////////

initialization
finalization
  FreeAndNil(fAvailableRegistries);

end.
